
//===============================================================================
//-------------------------------------------------------------------------------
//                           	Visual Framework Plug-in
//								version 1.0.0
//								Mike Wright
//								darwin@mbay.net
//
//  This is a sample module that is designed to simplify the writing of a
//	visual plugin for SoundJam. It includes the code needed to register the 
//	plugin with the app, as well as the code for the dispatching of incoming 
//	messages from SoundJam to the appropriate handler routines.
//
//	The purpose of this code is to help clarify how a visual plugin module 
//	should interact with SoundJam, not to teach how to create graphic displays,
//	nor how to process sound data.
//
//	There are also message-handler functions for each of the possible messages.
//	These functions are commented to make it easier to understand what kinds of
//	tasks they are expected to carry out.
//
//	A few of the functions have been filled out with enough code to produce a
//	plugin that actually does something--but not much. This plugin simply
//	does some very basic processing of a portion of the spectrum data
//	in order to produce some simple visual effects.
//
//	The handling of a pop-up menu, which comes up when the module window
//	is clicked, is also implemented.
//
//	There is a global data structure, the VisHandlerData struct, which 
//	contains the data which the plugin uses to do its jobs. This struct
//	should be modified to contain the data fields that your plugin will
//	need to do its job. Many of them will be the same as those in the sample
//	plugin module.
//
//	This information is specific to visual plug-ins, but please note that a
//	SoundJam plugin container may contain multiple plug-ins. In main(), when
//	registering the plug-n with SoundJam, the visual plugin passes info back
//	by way of the u.registerVisualPluginMessage element of the 
//	PlayerMessageInfo union struct. This union also includes the following
//	registration elements:
//		registerPlayerPluginMessage;
//		registerDSPPluginMessage;
//		registerEncoderPluginMessage;
//		registerSkinPluginMessage;
//	See the PlayerAPI.h header file for the details of the PlayerMessageInfo
//	struct.
//
//	Note that this sample module does not do much in the way of error checking,
//	but a serious plugin should be just as careful about error checking and
//	memory handling as any application.
//
//-------------------------------------------------------------------------------
//===============================================================================


//===============================================================================
//-------------------------------------------------------------------------------
//								 VisFrameworkMain.c
//
//	Handles calls from SoundJam.
//
//	main() 			handles registering our plugin module with SoundJam
//	VisHandler() 	retrieves our global data from the refcon and calls handlers
//				 for each SoundJam message
//
//-------------------------------------------------------------------------------
//===============================================================================


#include "Debug.h"
#include "VisFramework.h"


//========================================================================== Functions

/*  VisHandler 
Purpose:	This is the entry point where SoundJam will first call us. 

			We use the u.registerVisualPluginMessage of the PlayerMessageInfo 
			struct to register the visual plugin with SoundJam.
			
			It receives the following parameters:
			
				"messageInfo", which is a pointer to a union struct. Each struct
				contains fields that are required to handle the task indicated
				by the message.

				"messageType", which is a key to the struct element of the union 
				that is contained in messageInfo.
			
				"refcon", which is used to store the plugin's data structure.
				In this framework sample, the data structure type is the 
				VisHandlerData struct, which is defined in the VisFramework.h 
				header.
			
			The VisHandlerData struct should be redefined to hold whatever data
			your plugin will need access to at each call. This will include any
			data passed to the plugin via the "messageInfo" param.
			
			The PlayerMessageInfo struct, playerMessageInfo, is used to pass
			information about the plugin to SoundJam. The fields are filled in
			below according to the needs of the framework sample plugin, but you
			will want to fill them in to reflect the needs of your plugin.
			
			The values that can be used for the options field are as follows:
			
			  kVisualRequiresFullScreen		When the ration of the values for
											minHeight and minWidth match the
											ratio of height and width of the
											screen resolution, the plugin
											window will open to full screen.
												
			  kVisualForBuiltIn				NEVER set this bit. It is only
											for use by the built-in visual
											plugin (though it will probably
											just be ignored).
												
			  kVisualIsResizeable			The plugin window will be resizable,
											so you should be prepared to deal
											with resize messages. HOWEVER, this
											feature is NOT implemented in
											SoundJam 1.0.0.
												
			  kVisualWantsIdleMessages		The plugin will receive idle messages,
											at which time you may do whatever
											you wish.
												
			  kVisualWantsToProcessSamples	The plugin will receive processSamples
											messages, which will include a pointer
											to a sound buffer which can be
											modified by the plugin.
											
			  kVisualWantsConfigure			The plugin will receive configure
											messages. This is NOT enabled in
											SJ 1.0.0.
			
			  kVisualWantsAbout				The plugin will receive a message
											telling it to display its "About"
											information. This is NOT enabled in
											SJ 1.0.0.
			
			  kVisualEnabledByDefault		The plugin will be enabled when
											SoundJam is first run, without having
											to be turned on in the preferences.

Arguments:	messageType - kind of message contained in messageInfo struct
			messageInfo	-	pointer to a VisualPluginMessageInfo struct
			refcon		-	stored plugin global data

Returns:	noErr (always)

*/
OSStatus main (OSType messageType,PluginMessageInfo *messageInfo,void *refcon)
{
#pragma unused (refcon)
	PlayerMessageInfo	playerMessageInfo;		/* struct for passing our info back to SoundJam */
	Str255				plugInName			= "\pWhiteCap";	/* put the name of your plugin here */
	OSStatus			status				= noErr;
	
	// Look at message passed in to us.
	switch (messageType)
	{
		case kPluginInitializeMessage:
			// Fill in playerMessageInfo fields for passing to PlayerRegisterVisualPlugin()
			BlockMove((Ptr)&plugInName[0],(Ptr)&playerMessageInfo.u.registerVisualPluginMessage.name[0],plugInName[0] + 1);
			playerMessageInfo.u.registerVisualPluginMessage.options = kVisualEnabledByDefault | kVisualIsResizeable | kVisualWantsIdleMessages;
			playerMessageInfo.u.registerVisualPluginMessage.messageVersion = kCurrentPluginMessageVersion;
			playerMessageInfo.u.registerVisualPluginMessage.handler = VisHandler;
			playerMessageInfo.u.registerVisualPluginMessage.registerRefcon = 0;
			
			// Specify timing/data information.
			playerMessageInfo.u.registerVisualPluginMessage.timeBetweenDataInMS = 1;
			playerMessageInfo.u.registerVisualPluginMessage.numWaveformChannels = 0;
			playerMessageInfo.u.registerVisualPluginMessage.numSpectrumChannels = 2;
			
			// Specify our bounds constraints.
			playerMessageInfo.u.registerVisualPluginMessage.minWidth = 300;		// Backward compat: put default win size here
			playerMessageInfo.u.registerVisualPluginMessage.minHeight = 200;
			playerMessageInfo.u.registerVisualPluginMessage.maxWidth = 640;		// Backward compat: put fullscreen size here
			playerMessageInfo.u.registerVisualPluginMessage.maxHeight = 480;
			
			playerMessageInfo.u.registerVisualPluginMessage.windowAlignmentInBytes = 0;
			
			// Call the player.
			status = PlayerRegisterVisualPlugin(messageInfo->u.initMessage.appCookie,
					messageInfo->u.initMessage.playerProc,&playerMessageInfo);


		break;
		
		case kPluginCleanupMessage:
		break;
		
		default:
			DebugMsg("\pVisHandler: Unknown message number 1.");	
			status = paramErr;
		break;
	}
			
	return status;
	
} /* end main */



/*  VisHandler 
Purpose:	This is the function that handles messages from SoundJam.
			It was registered with SoundJam in main().
			
			It receives the following parameters:
			
				"messageInfo", which is a pointer to a union struct. Each struct
				contains fields that are required to handle the task indicated
				by the message.

				"messageType", which is a key to the struct element of the union 
				that is contained in messageInfo.
			
				"refcon", which is used to store the plugin's data structure.
				In this framework sample, the data structure type is the 
				VisHandlerData struct, which is defined in the VisFramework.h 
				header.
			
			The VisHandlerData struct should be redefined to hold whatever data
			your plugin will need access to at each call. This will include any
			data passed to the plugin via the "messageInfo" param.
			
			Each messageType has a matching handler that is called by VisHandler().
			Future versions of SoundJam will probably add more message types.

Arguments:	messageType - 	kind of message contained in messageInfo struct
			messageInfo	-	pointer to a VisualPluginMessageInfo struct
			refcon		-	stored plugin global data (VisHandlerData)

Returns:	status

*/
OSStatus VisHandler (OSType messageType,VisualPluginMessageInfo *messageInfo,void *refcon)
{
	VisHandlerData *plugInData;
	OSStatus		status;
	
	// Assume no errors.
	status = noErr;
	
	// Get a grab on our plugin data.
	plugInData = (VisHandlerData *)refcon;
	
	// See which message we're being sent and call appropriate function in VISPluginHandlers.c.
	switch (messageType)
	{
		case kVisualPluginInitializeMessage:
		status = HandleInitializeMessage(messageInfo,plugInData);		// first time in, setting up
		break;
		
		case kVisualPluginCleanupMessage:
		status = HandleCleanupMessage(plugInData);						// shutting down, disposing allocated memory
		break;
		
		case kVisualPluginEnableMessage:
		status = HandleEnableMessage(plugInData);						// turn on the module, allocate more memory
		break;
		
		case kVisualPluginDisableMessage:
		status = HandleDisableMessage(plugInData);						// turn off the module, dispose memory allocated in HandleEnableMessage
		break;
		
		case kVisualPluginShowWindowMessage:
		status = HandleShowWindowMessage(messageInfo,plugInData);		// the plugin window has been made visible
		break;
		
		case kVisualPluginHideWindowMessage:
		status = HandleHideWindowMessage(plugInData);					// the plugin window has been hidden
		break;
		
		case kVisualPluginSetWindowMessage:
		status = HandleSetWindowMessage(messageInfo,plugInData);		// the plugin window has been changed
		break;
		
		case kVisualPluginResizeMessage:
		status = HandleResizeMessage(messageInfo,plugInData);			// user has resized the plugin window
		break;
		
		case kVisualPluginPlayMessage:
		status = HandlePlayMessage(messageInfo,plugInData);				// a track has started playing
		break;
		
		case kVisualPluginStopMessage:
		status = HandleStopMessage(plugInData);							// the current track has stopped playing
		break;
		
		case kVisualPluginPauseMessage:
		status = HandlePauseMessage(plugInData);						// the current track has been paused
		break;
		
		case kVisualPluginUnpauseMessage:
		status = HandleUnpauseMessage(plugInData);						// the current track has been unpaused
		break;
		
		case kVisualPluginRenderMessage:
		status = HandleRenderMessage(messageInfo,plugInData);			// render some some samples as visual
		break;
		
		case kVisualPluginProcessSamplesMessage:
		status = HandleProcessSamplesMessage(messageInfo,plugInData);	// process (modify) sound samples
		break;
		
		case kVisualPluginFlushSamplesMessage:
		status = HandleFlushSamplesMessage(plugInData);					// dispose old sound samples and be ready for new ones
		break;
		
		case kVisualPluginEventMessage:
		status = HandleEventMessage(messageInfo,plugInData);			// handle clicks and keydowns in module window
		break;
		
		case kVisualPluginUpdateMessage:
		status = HandleUpdateMessage(plugInData);						// handle updates of module window
		break;
		
		case kVisualPluginIdleMessage:
		status = HandleIdleMessage(plugInData);							// the idle time you requested is available
		break;
		
		default:
		DebugMsg("\pVisHandler: Unknown message number 2.");			// What could the programmer be thinking?
		status = paramErr;
		break;
	}
	
bail:
			
	return status;

} /* end VisHandler */

